package com.agisoft.metashape;

import java.util.Optional;

/**
 * Transformation between chunk and world coordinates systems.
 */
public class Transform {

	private Matrix rotation;
	private Vector translation;
	private Double scale;

	public Transform() {
		reset();
	}

	/**
	 *  Rotation component, may be null.
	 */
	public void setRotation(Matrix rotation) {
		this.rotation = rotation;
	}

	/**
	 *  Rotation component, may be null.
	 */
	public Optional<Matrix> getRotation() {
		return Optional.ofNullable(rotation);
	}

	/**
	 *  Translation component, may be null.
	 */
	public void setTranslation(Vector translation) {
		this.translation = translation;
	}

	/**
	 *  Translation component, may be null.
	 */
	public Optional<Vector> getTranslation() {
		return Optional.ofNullable(translation);
	}

	/**
	 *  Scale component, may be null.
	 */
	public void setScale(Double scale) {
		this.scale = scale;
	}

	/**
	 *  Scale component, may be null.
	 */
	public Optional<Double> getScale() {
		return Optional.ofNullable(scale);
	}

	/**
	 *  Transformation matrix.
	 */
	public void setMatrix(Matrix matrix) {
		translation = matrix.translation();
		rotation = matrix.rotation();
		scale = matrix.scale();
	}

	/**
	 *  Transformation matrix.
	 */
	public Matrix getMatrix() {
		Vector translation = this.translation != null ? this.translation : new Vector(0, 0, 0);
		Matrix rotation = this.rotation != null ? this.rotation : Matrix.makeDiag(new Vector(1, 1, 1));
		Double scale = this.scale != null ? this.scale : 1.0;

		return Matrix.makeTranslation(translation).mul(Matrix.makeRotation(rotation)).mul(Matrix.makeScale(new Vector(scale, scale, scale)));
	}

	public boolean hasMatrix() {
		return translation != null && rotation != null && scale != null;
	}

	public void reset() {
		translation = null;
		rotation = null;
		scale = null;
	}

	public Vector transformPoint(Vector point) {
		return getMatrix().mulp(point);
	}

	public Vector transformVector(Vector vector) {
		return getMatrix().mulv(vector);
	}

}
